#!/usr/bin/env julia

using ArgParse

import Viterbi
import PyInterface

trainingfiles = [
	("genomes/genome1.fa", "annotations/annotation1.fa"),
	("genomes/genome2.fa", "annotations/annotation2.fa"),
	("genomes/genome3.fa", "annotations/annotation3.fa"),
	("genomes/genome4.fa", "annotations/annotation4.fa"),
	("genomes/genome5.fa", "annotations/annotation5.fa")
	]

function annotatefile(trainingfiles, model, file, output)
	modelJson = PyInterface.parseModel(model)

	hmm = PyInterface.createHMM(modelJson)

	for pair in trainingfiles
		PyInterface.trainByCounting(hmm, pair)
	end

	PyInterface.normalize(hmm)

	dna = PyInterface.readFasta(file)
	w = Viterbi.run(hmm, dna)

	track = Viterbi.backtrack(hmm, dna, w)
	Viterbi.annotate(hmm, track, output)
end

function crossvalidate(trainingfiles, model, output)
	for i = 1:length(trainingfiles)
		trainset = Array((ASCIIString,ASCIIString), 0)
		testpair  = ("", "")

		for (j, pair) in enumerate(trainingfiles)
			if j != i
				push!(trainset, pair)
			else
				testpair = pair
			end
		end

		println("Testing on $(testpair[1]):")

		# Annotate the testfile
		Viterbi.annotatefile(trainset, model, testpair[1], output)

		# Generate annotation outpu
		run(`./compare_anns.py $(testpair[2]) $(output)`)
		println("")
	end
end

function main()
	s = ArgParseSettings()
	@add_arg_table s begin
		"model"
			help = "use the model for predicting"
			arg_type = String
			required = true
		"--crossvalidation", "-c"
			help = "enable cross validation mode"
			action = :store_true
		"--annotate", "-a"
			help = "creates an annotation of the argument file, training on already annotated genomes"
			arg_type = String
			default  = ""
		"--output", "-o"
			help = "output annotation to give file"
			arg_type = String
			default  = "annotated"
	end

	args = parse_args(s)

	if args["annotate"] != ""
		annotatefile(trainingfiles, args["model"], args["annotate"], args["output"])
	end

	if args["crossvalidation"]
		crossvalidate(trainingfiles, args["model"], args["output"])
	end
end

main()
