#!/usr/bin/python3

import sys
import json
import numpy

from subprocess import call

def toDot(hmm, outputfile):
    output = "model.dot"
    
    #              BLUEISH    REDISH     ORANGE
    colors = set(["#1E88E5", "#F44336", "#FB8C00"])
    usedColors = dict()

    try:
        f = open(output, "w")

        f.write("digraph {\n\trankdir=LR;\n")
        f.write("\tnode [fontname=\"Palatino\", fontsize=\"16\", style=\"filled\"];\n\n")

        for k, v in hmm['states'].items():

            f.write("\t\"" + k + "\" [label=\"{0}\\n".format(k))

            # Write probability of emission
            for a, i in hmm['aIndex'].items():
                val = hmm['phi'][v['i']][i]*100
                if val > 0:
                        f.write("{0} = {1:.1f}%\\n".format(a, val))

            f.write("\", ")

            if v['a'] in usedColors:
                c = usedColors[v['a']]
            else:
                c = colors.pop()
                usedColors[v['a']] = c

            f.write("fillcolor=\"{0}\"".format(c))
            f.write("];\n")

        for k, v in hmm['states'].items():
            for n in v['t']:
                previndex =v['i']
                curindex = hmm['states'][n]['i']
                val = hmm['A'][previndex][curindex]*100

                if val >= 1:
                    f.write("\t\"{0}\" -> \"{1}\" [label=\"{2:.1f}%\"];\n".format(k, n, val))
                else:
                    f.write("\t\"{0}\" -> \"{1}\" [label=\"{2:.4f}%\"];\n".format(k, n, val))

        f.write("}")

        f.close()
    except IOError as e:
        print("I/O error occured:" + e)
        sys.exit(1)

    call(('dot -Tpng -o ' + outputfile + ' model.dot').split())

def readJsonFile(i):
    try:
        f = open(i, 'r')
        j = json.load(f)
    except IOError as e:
        print ("I/O error({0}) on '{2}': {1}").format(e.errno, e.strerror, i)
        sys.exit()

    return j

def createHMM(j):
    HMM = dict()

    alphabet = set()
    # Scan through emission_probabilities and default_emissions to get alphabet
    for k in j['default_emissions'].keys():
        if k not in alphabet:
            alphabet.add(k)

    #for k in j['emission_probabilities'].values():
    #    for t in k:
    #        if t not in alphabet:
    #            alphabet.add(t)

    states = dict()
    stateSet = set() # Used to initially add all states

    emissions   = j['emission_probabilities']
    annotations = j['annotation']
    transitions = j['transitions']

    # Allow adding single output path easily by adding them to emission_probabilities.
    splitNodes = set()
    redirectTable = dict()

    emcopy = emissions.copy()
    for k, v in emissions.items():
        ann = annotations[k]

        names = list()
        i = 0
        for symbols in v: # All but last
            i += 1
            name = "{0}({1}:{2})".format(k, i, symbols)
            names.append(name)
            emcopy[name]   = symbols
            annotations[name] = ann
            stateSet.add(name)


        # The first elements is the one others should "go" to instead of k
        redirectTable[k] = names[0]

        prev = names[0]
        for n in names[1:]:
            transitions[prev] = [n]
            prev = n

        # The last element in the list should do the state hopping
        last = names[-1:][0]
        transitions[last] = transitions[k]

        if k in emcopy:
            emcopy.pop(k)
        if k in annotations:
            annotations.pop(k)
        if k in transitions:
            transitions.pop(k)

    emissions = emcopy


    for k in transitions.keys():
        stateSet.add(k)

    for k in annotations.keys():
        stateSet.add(k)

    # The index count of our states into the A and Φ tables.
    statecount = 0

    # Add all states to 'states', along with what states they can transition
    # to.
    for k in stateSet:
        states[k] = {}

        # Add all transitions to the state we just found
        if k in transitions:
            t = transitions[k]
            trans = []
            for v in t:
                if v not in redirectTable:
                    trans.append(v)
                else:
                    trans.append(redirectTable[v])

            states[k]['t'] = trans

        if k in annotations:
            states[k]['a'] = annotations[k]

        states[k]['i'] = statecount
        statecount += 1


    HMM['states'] = states

    start = dict()
    for k, v in j['annotation_start'].items():
        if v not in redirectTable:
            start[k] = v
        else:
            start[k] = redirectTable[v]

    HMM['start']  = start

    HMM['static'] = emissions

    # The A table store the probability to go from state i to j in [i][j]
    HMM['A']      = numpy.zeros(shape=(len(states), len(states)), dtype=numpy.float64)

    # Create the Φ table that stores emissions per possible state
    HMM['phi']    = numpy.zeros(shape=(len(states), len(alphabet)), dtype=numpy.float64)

    # Create the index map for our alphabet
    aindex = dict()
    i = 0
    for a in alphabet:
        aindex[a] = i
        i += 1
    HMM['aIndex'] = aindex

    return HMM

if __name__ == "__main__":
    j = readJsonFile('v1.json')
    HMM = createHMM(j)
    print(HMM)
