#define _DEFAULT_SOURCE

#include "streams.h"
#include "stream_common.h"
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/types.h>


typedef struct input {
	int fd;
	uint64_t end;
	uint64_t cur;
	void *map;
	int map_size;
	uint64_t bp;
	uint64_t B;
	uint64_t bpOffset;
	uint64_t offset;
} input_stream;

typedef struct output {
	int fd;
	uint64_t end;
	uint64_t cur;
	void *map;
	int map_size;
	uint64_t bp;
	uint64_t B;
	uint64_t bpOffset;
	uint64_t offset;
} output_stream;

static input_stream **in_list;
static int in_len;

static output_stream **out_list;
static int out_len;

__attribute__ ((visibility("default")))
int mi_open(const char *f, uint64_t start, uint64_t length, int B) {
	int id = 0;
	long page_size;

	id = stream_find_struct_storage((uintptr_t **)&in_list, &in_len);

	input_stream *s =  xmalloc(sizeof(input_stream));
	in_list[id] = s;

	s->fd = open(f, O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR);

	if(s->fd < 0) {
		printf("Error in mi_open: '%s'\n", strerror(errno));
		exit(1);
	}

	s->end = sizeof(uint32_t) * (start + length);
	s->cur = sizeof(uint32_t) * start;
	s->bp = 0;
	s->B = B;

	page_size = sysconf(_SC_PAGE_SIZE);
	s->bpOffset = s->cur % page_size;
	s->offset = s->cur - s->bpOffset;
	s->map_size = sizeof(uint32_t) * B + s->bpOffset;

	s->map = mmap(NULL, s->map_size, PROT_READ, 
			MAP_SHARED, s->fd, s->offset);

	if (s->map == MAP_FAILED) {
		printf("Unable to map@open: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	return id;
}

__attribute__ ((visibility("default")))
uint32_t mi_next(int id) {
	int res;
	long page_size;

	input_stream *i = in_list[id];

	if (i->bp == i->B) {
		// Unmap earlier mapped memory.
		res = munmap(i->map, i->map_size);
		if (res == -1) {
			printf("Unable to unmap@next: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		// Mmap needs a size aligned with PAGE_SIZE.
		page_size = sysconf(_SC_PAGE_SIZE);

		// bpOffset is buffer offset into the nearest aligned page.
		i->bpOffset = i->cur % page_size;

		// offset is the beginning of the aligned page.
		i->offset = i->cur - i->bpOffset;

		// If our buffer is bigger than the remaining things we need to read.
		if (i->end - i->cur < sizeof(uint32_t) * i->B) {
			// Just allocate a size that corresponds to the rest of the file.
			i->map_size = i->end - i->cur + i->bpOffset;
		} else {
			// Else allocate the size of our buffer.
			i->map_size = sizeof(uint32_t) * i->B + i->bpOffset;
		}

		i->map = mmap(NULL, i->map_size, PROT_READ, MAP_PRIVATE, i->fd, 
				i->offset);

		if (i->map == MAP_FAILED) {
			printf("Unable to map@next: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		i->bp = 0;
	}

	i->cur += sizeof(uint32_t);

	uint32_t *valindex = (uint32_t *)((char *)i->map
		+ i->bpOffset 
		+ (i->bp * sizeof(uint32_t)));

	i->bp++;

	return *valindex;
}

__attribute__ ((visibility("default")))
int mi_eos(int id) {
	input_stream *i = in_list[id];
	if(i->cur >= i->end) {
		return 1;
	}

	return 0;
}

__attribute__ ((visibility("default")))
void mi_close(int id) {
	int res;
	input_stream *i = in_list[id];

	res = munmap(i->map, i->map_size);

	if (res == -1) {
		printf("Unable to unmap@close: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	close(i->fd);

	free(i);

	in_list[id] = 0;
}


__attribute__ ((visibility("default")))
int mo_create(const char *f, uint64_t start, uint64_t length, int B) {
	int od = 0;
	int r;
	long page_size;
	struct stat stat_buf;

	od = stream_find_struct_storage((uintptr_t **)&out_list, &out_len);

	output_stream *o = xmalloc(sizeof(output_stream));
	out_list[od] = o;

	o->fd = open(f, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);

	if(o->fd < 0) {
		printf("Error in mo_create: '%s'\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	r = fstat(o->fd, &stat_buf);

	if (r == -1) {
		printf("Unable to stat in mo_create: '%s'\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	o->end = sizeof(uint32_t) * (start + length);
	o->cur = sizeof(uint32_t) * start;
	o->B = B;
	o->bp = 0;

	// Expand the file so we can write to int.
	if ((uint64_t)stat_buf.st_size < o->end) {
		r = ftruncate(o->fd, o->end);

		if (r == -1) {
			printf("Unable to truncate in mo_create: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}
	}

	page_size = sysconf(_SC_PAGE_SIZE);
	o->bpOffset = o->cur % page_size;
	o->offset = o->cur - o->bpOffset;
	o->map_size = sizeof(uint32_t) * B + o->bpOffset;

	o->map = mmap(NULL, o->map_size, PROT_WRITE, 
			MAP_SHARED, o->fd, o->offset);

	if (o->map == MAP_FAILED) {
		printf("Unable to map@create: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	return od;
}

__attribute__ ((visibility("default")))
void mo_write(int od, uint32_t v) {
	output_stream *o = out_list[od];
	int res;
	long page_size;

	if (o->bp == o->B) {
		res = munmap(o->map, o->map_size);

		if (res == -1) {
			printf("Unable to unmap@write: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		page_size = sysconf(_SC_PAGE_SIZE);
		o->bpOffset = o->cur % page_size;
		o->offset = o->cur - o->bpOffset;

		// If our buffer is bigger than the remaining things we need to read.
		if (o->end - o->cur < sizeof(uint32_t) * o->B) {
			// Just allocate a size that corresponds to the rest of the file.
			o->map_size = (o->end - o->cur) + o->bpOffset;
		} else {
			// Else allocate the size of our buffer.
			o->map_size = sizeof(uint32_t) * o->B + o->bpOffset;
		}


		o->map = mmap(NULL, o->map_size, PROT_WRITE, MAP_SHARED, o->fd, 
				o->offset);

		if (o->map == MAP_FAILED) {
			printf("Unable to mmap@write: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		o->bp = 0;
	}

	uint32_t *values = (uint32_t *)((char *)o->map + o->bpOffset);
	values[o->bp] = v;
	o->bp++;

	o->cur += sizeof(uint32_t);
}

__attribute__ ((visibility("default")))
void mo_close(int od) {
	output_stream *o = out_list[od];
	int res;

	res = munmap(o->map, o->map_size);

	if (res == -1) {
		printf("Unable to unmap@close: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	close(o->fd);

	free(o);

	out_list[od] = 0;
}
