#include "streams.h"
#include "stream_common.h"
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

typedef struct input {
	FILE *fd;
	uint64_t end;
	uint64_t cur;
} input_stream;

typedef struct output {
	FILE *fd;
} output_stream;

input_stream **in_list;
int in_len = 0;

output_stream **out_list;
int out_len = 0;

__attribute__ ((visibility("default")))
int fi_open(const char *f, uint64_t start, uint64_t length) {
	int id = 0;
	int res;
	struct stat stat_buf;

	id = stream_find_struct_storage((uintptr_t **)&in_list, &in_len);

	input_stream *s =  xmalloc(sizeof(input_stream));
	in_list[id] = s;

	res = stat(f, &stat_buf);
	if (res == -1) {
		switch errno {
			case ENOENT:
				s->fd = fopen(f, "w+");
				break;
			default:
				printf("Unable to open output: %s\n", strerror(errno));
				exit(EXIT_FAILURE);
		}
	} else {
		s->fd = fopen(f, "r+");
	}

	if(s->fd == 0) {
		printf("Error in fi_open: '%s'\n", strerror(errno));
		exit(1);
	}

	s->cur = sizeof(uint32_t) * start;
	s->end = sizeof(uint32_t) * (start + length);

	fseek(s->fd, s->cur, SEEK_SET);

	return id;
}

__attribute__ ((visibility("default")))
uint32_t fi_next(int id) {
	int32_t res;
	input_stream *i = in_list[id];
	int r;
	r = fread(&res, 1, sizeof(uint32_t), i->fd);
	(void) r;

	if (ferror(i->fd) != 0) {
		errorAndDie(3);
	}

	i->cur += sizeof(uint32_t);

	return res;
}

__attribute__ ((visibility("default")))
int fi_eos(int id) {
	input_stream *i = in_list[id];
	if(i->cur >= i->end) {
		return 1;
	}

	return 0;
}

__attribute__ ((visibility("default")))
void fi_close(int id) {
	input_stream *i = in_list[id];
	fclose(i->fd);

	free(i);

	in_list[id] = 0;
}

__attribute__ ((visibility("default")))
int fo_create(const char *f, uint64_t start) {
	int od = 0;
	int res;
	struct stat stat_buf;

	od = stream_find_struct_storage((uintptr_t **)&out_list, &out_len);

	output_stream *o = xmalloc(sizeof(output_stream));
	out_list[od] = o;

	res = stat(f, &stat_buf);
	if (res == -1) {
		switch errno {
			case ENOENT:
				o->fd = fopen(f, "w+");
				break;
			default:
				printf("Unable to open output: %s\n", strerror(errno));
				exit(EXIT_FAILURE);
		}
	} else {
		o->fd = fopen(f, "r+");
	}

	if(o->fd == 0) {
		printf("Error in fo_create: '%s'\n", strerror(errno));
		exit(1);
	}

	fseek(o->fd, start*sizeof(uint32_t), SEEK_SET);

	return od;
}

__attribute__ ((visibility("default")))
void fo_write(int od, uint32_t v) {
	output_stream *o = out_list[od];
	int r;
	r = fwrite(&v, 1, sizeof(uint32_t), o->fd);
	(void) r;

	if(ferror(o->fd) != 0) {
		errorAndDie(4);
	}
}

__attribute__ ((visibility("default")))
void fo_close(int od) {
	output_stream *o = out_list[od];
	fflush(o->fd);
	fclose(o->fd);
	free(o);

	out_list[od] = 0;
}
