#include "streams.h"
#include "stream_common.h"
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

typedef struct input {
	int fd;
	uint64_t end;
	uint64_t cur;
	uint32_t *buf;
	uint64_t bp;
	unsigned int B;
} input_stream;

typedef struct output {
	int fd;
	uint32_t *buf;
	uint64_t bp;
	unsigned int B;
} output_stream;

static input_stream **in_list;
static int in_len;

static output_stream **out_list;
static int out_len;

__attribute__ ((visibility("default")))
int bi_open(const char *f, uint64_t start, uint64_t length, int B) {
	int id = 0;

	id = stream_find_struct_storage((uintptr_t **)&in_list, &in_len);

	input_stream *s =  xmalloc(sizeof(input_stream));
	in_list[id] = s;

	// Allocate the buffer on first use.
	s->buf = NULL;
	s->B = B;
	s->bp = B;
	s->fd = open(f, O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR);

	if(s->fd < 0) {
		printf("Error in bi_open: '%s'\n", strerror(errno));
		exit(1);
	}

	s->cur = sizeof(uint32_t) * start;
	s->end = sizeof(uint32_t) * (start + length);

	lseek(s->fd, s->cur, SEEK_SET);

	return id;
}

__attribute__ ((visibility("default")))
uint32_t bi_next(int id) {
	uint64_t r;
	int err;

	input_stream *i = in_list[id];

	// Allocate the buffer on first use.
	if (i->buf == NULL) {
		i->buf =  xcalloc(sizeof(uint32_t), i->B);
	}

	if (i->bp == i->B) {
		if (i->end - i->cur < i->B * sizeof(uint32_t)) {
			r = i->end - i->cur;
		} else {
			r = sizeof(uint32_t) * i->B;
		}

		err = read(i->fd, i->buf, r);
		if (err == -1) {
			errorAndDie(5);
		}
		i->bp = 0;
	}

	i->cur += sizeof(uint32_t);

	return i->buf[i->bp++];
}

__attribute__ ((visibility("default")))
int bi_eos(int id) {
	input_stream *i = in_list[id];
	if(i->cur >= i->end) {
		return 1;
	}

	return 0;
}

__attribute__ ((visibility("default")))
void bi_close(int id) {
	input_stream *i = in_list[id];
	close(i->fd);

	if (i->buf != NULL) {
		free(i->buf);
	}
	free(i);

	in_list[id] = 0;
}


__attribute__ ((visibility("default")))
int bo_create(const char *f, uint64_t start, int B) {
	int od = 0;

	od = stream_find_struct_storage((uintptr_t **)&out_list, &out_len);

	output_stream *o = xmalloc(sizeof(output_stream));
	out_list[od] = o;

	// Allocate buffer on first use.
	o->buf = NULL;
	o->B = B;
	o->bp = 0;
	o->fd = open(f, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);

	if(o->fd < 0) {
		printf("Error in bo_create: '%s'\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	lseek(o->fd, start*sizeof(uint32_t), SEEK_SET);

	return od;
}

__attribute__ ((visibility("default")))
void bo_write(int od, uint32_t v) {
	output_stream *o = out_list[od];
	int r;

	if (o->buf == NULL) {
		o->buf =  xcalloc(sizeof(uint32_t), o->B);
	}

	o->buf[o->bp] = v;
	o->bp++;

	if (o->bp == o->B) {
		r = write(o->fd, o->buf, sizeof(uint32_t) * o->B);

		if(r == -1) {
			errorAndDie(6);
		}

		o->bp = 0;
	}
}

__attribute__ ((visibility("default")))
void bo_close(int od) {
	output_stream *o = out_list[od];
	int r;

	// bp is above zero only if buf is initialized.
	if (o->bp > 0) {
		r = write(o->fd, o->buf, sizeof(uint32_t) * o->bp);

		if(r == -1) {
			errorAndDie(7);
		}
	}

	close(o->fd);

	if (o->buf != NULL) {
		free(o->buf);
	}

	free(o);

	out_list[od] = 0;
}
