#define _GNU_SOURCE

#include <inttypes.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <time.h>

#include "../xutil.h"
#include "../matrix.h"
#include "../naive.h"
#include "../naive_idx.h"
#include "../frigo.h"
#include "../drepper.h"
#include "../strassen.h"
#include "../strassen_idx.h"

void printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-m rows of A]\n"
			"\t[-n cols of A]\n"
			"\t[-p cols of B]\n"
			"\t[-s seed]\n"
			, argv[0]);
}

int main(int argc, char *argv[]) {
	int opt;
	uint32_t m = 0, n = 0, p = 0;
	uint32_t seed = 0;
	struct matrix_mult str;
	struct strassen_mult str_str;
	struct timespec begin, end;
	int total_ns, total_s;

	(void)str;
	(void)str_str;

	matrix_t *A, *B, *C;

	while ((opt = getopt(argc, argv, "s:m:n:p:")) != -1) {
		switch (opt) {
		case 'm':
			m = strtoull(optarg, NULL, 10);
			break;
		case 'n':
			n = strtoull(optarg, NULL, 10);
			break;
		case 'p':
			p = strtoull(optarg, NULL, 10);
			break;
		case 's':
			seed = strtoull(optarg, NULL, 10);
			break;
		default:
			printusage(argv);
			exit(EXIT_FAILURE);
		}
	}

	if (!seed) {
		srand(seed);
	}

	if (!m || !n || !p) {
		printusage(argv);
		exit(EXIT_FAILURE);
	}

	printf(
			"Testing on A(%"PRIu32"x%"PRIu32") * B(%"PRIu32"x%"PRIu32") = " 
			"C(%"PRIu32"x%"PRIu32")\n", m, n, n, p, m, p
	);

	A = matrix_create(m, n);
	B = matrix_create(n, p);
	C = matrix_create(m, p);

	matrix_randomize(A, (unsigned int)rand());
	matrix_randomize(B, (unsigned int)rand());

	str.A = A;
	str.B = B;
	str.C = C;

	str_str.A = A;
	str_str.B = B;
	str_str.C = C;
	str_str.th = 0;

	clock_gettime(CLOCK_REALTIME, &begin);

	//naive(&str);
	frigo(&str);
	//drepper(&str);
	//strassen(&str_str);

	clock_gettime(CLOCK_REALTIME, &end);

	if (begin.tv_nsec <= end.tv_nsec) {
		total_ns = end.tv_nsec - begin.tv_nsec;
		total_s  = end.tv_sec  - begin.tv_sec;
	} else {
		total_ns = end.tv_nsec + (1e9 - begin.tv_nsec);
		total_s  = end.tv_sec - begin.tv_sec - 1;
	}

	printf("%f\n", total_s + total_ns/1e9);

	matrix_destroy(A);
	matrix_destroy(B);
	matrix_destroy(C);

	return EXIT_SUCCESS;
}
