# Multiple plot function
#
# ggplot objects can be passed in ..., or to plotlist (as a list of ggplot objects)
# - cols:   Number of columns in layout
# - layout: A matrix specifying the layout. If present, 'cols' is ignored.
#
# If the layout is something like matrix(c(1,2,3,3), nrow=2, byrow=TRUE),
# then plot 1 will go in the upper left, 2 will go in the upper right, and
# 3 will go all the way across the bottom.
#

POINTSIZE <- 0.9
LINESIZE  <- 0.3

contains <- function(ar, d) {
	for (e in ar) {
		if (e == d) {
			return(TRUE)
		}
	}

	return(FALSE)
}

multiplot <- function(plots, file, cols=1, layout=NULL) {
  require(grid)

  # Make a list from the ... arguments and plotlist
  #plots <- c(list(...), plotlist)

  numPlots = length(plots)

  # If layout is NULL, then use 'cols' to determine layout
  if (is.null(layout)) {
    # Make the panel
    # ncol: Number of columns of plots
    # nrow: Number of rows needed, calculated from # of cols
    layout <- matrix(seq(1, cols * ceiling(numPlots/cols)),
                    ncol = cols, nrow = ceiling(numPlots/cols))
  }

 if (numPlots==1) {
    print(plots[[1]])

  } else {
    # Set up the page
    grid.newpage()
    pushViewport(viewport(layout = grid.layout(nrow(layout), ncol(layout))))

    # Make each plot, in the correct location
    for (i in 1:numPlots) {
      # Get the i,j matrix positions of the regions that contain this subplot
      matchidx <- as.data.frame(which(layout == i, arr.ind = TRUE))

      print(plots[[i]], vp = viewport(layout.pos.row = matchidx$row,
                                      layout.pos.col = matchidx$col))
    }
  }
}

dae_magic <- function(x) {
	v <- load_csv(x);
	t <- dae_summarize(v);
	dae_plot(t);

	return(t);
}

load_csv <- function(x) {
	require("data.table")
	d <- read.csv(header=TRUE, sep=',', file=x);
	d <- data.table(d)
	return(d);
}

dae_summarize <- function(d) {
	new <- d[, lapply(.SD, mean), by = list(Testfile, Testsize)]

	return(new)
}

dae_l1_data_cache_misses <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=ifelse(!PAPI_L1_DCA, 0, 100*(PAPI_L1_DCM/PAPI_L1_DCA)), colour=Testfile, shape=Testfile, linetype=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="%", title="L1 Data Cache misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_l1_data_cache_misses_count <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=PAPI_L1_DCM, colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="#", title="L1 Data Cache misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_l1_instruction_cache_misses <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=ifelse(!PAPI_L1_ICA, 0, 100*(PAPI_L1_ICM/PAPI_L1_ICA)), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="%", title="L1 Instruction Cache misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_l3_cache_misses <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=ifelse(!PAPI_L3_TCA, 0, (100*(PAPI_L3_TCM/PAPI_L3_TCA))), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="%", title="L3 Cache misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_l3_cache_misses_count <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=PAPI_L3_TCM, colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="#", title="L3 Cache misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_l2_cache_accesses <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=PAPI_L2_DCA, colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="#", title="L2 Cache Accesses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_l2_cache_misses <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=ifelse(!PAPI_L2_DCA, 0, 100*(PAPI_L2_DCM/PAPI_L2_DCA)), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="%", title="L2 Cache misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_l2_cache_misses_count <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=PAPI_L2_DCM, colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="#", title="L2 Cache misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_branch_percent_mispredictions <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=ifelse(!PAPI_BR_CN, 0, 100*(PAPI_BR_MSP/PAPI_BR_CN)), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="%", title="Percentage Branch Mispredictions")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_branch_mispredictions <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=PAPI_BR_MSP, colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="#", title="Branch Mispredictions")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_running_time <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=((1e9*Seconds+Nanoseconds)/1e9), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="Seconds", title="Running Time")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_instructions_completed <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=log(PAPI_TOT_INS), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="#", title="Instructions Completed")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_tlb_data_misses <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=PAPI_TLB_DM, colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="#", title="TLB Data misses")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_percentage_cycles_stalled <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=ifelse(!PAPI_TOT_CYC, 0, 100*(PAPI_RES_STL/PAPI_TOT_CYC)), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="%", title="Percentage of cycles stalled")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_fraction_completed_to_issued <- function(d) {
	p <- ggplot(d, aes(x=log2(Testsize), y=ifelse(!PAPI_TOT_IIS, 0, PAPI_TOT_INS/PAPI_TOT_IIS), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="Fraction", title="Completed/Issued Instructions")
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + theme(legend.title=element_blank())

	return(p)
}

dae_plot <- function(d, filter=c(), select=c()) {
	require("ggplot2")

	for (i in filter) {
		d <- d[which(d$Testfile != i),]
	}

	dsave <- data.frame()
	for (i in select) {
		dsave <- rbind(dsave, d[which(d$Testfile == i),])
	}

	if (length(select) > 0) {
		d <- dsave
	}

	#d <- d[which(d$Testsize > 2^13 & d$Testsize <= 2^27),]
	#d = d[which(d$Testsize > 2^13),]

	plots <- list()

	plots <- c(plots, list(dae_l1_data_cache_misses(d)))
	plots <- c(plots, list(dae_l1_data_cache_misses_count(d)))
	plots <- c(plots, list(dae_l1_instruction_cache_misses(d)))

	if (contains(names(d), 'PAPI_L2_DCA')) {
		if (sum(d$PAPI_L2_DCA) != 0) {
			plots <- c(plots, list(dae_l2_cache_accesses(d)))

			if (contains(names(d), 'PAPI_L2_DCM')) {
				if (sum(d$PAPI_L2_DCM) != 0) {
					plots <- c(plots, list(dae_l2_cache_misses(d)))
					plots <- c(plots, list(dae_l2_cache_misses_count(d)))
				}
			}
		}
	}

	if (contains(names(d), 'PAPI_L3_TCM') && contains(names(d), 'PAPI_L3_TCA')) {
		if (sum(d$PAPI_L3_TCM) != 0 && sum(d$PAPI_L3_TCA) != 0) {
			plots <- c(plots, list(dae_l3_cache_misses(d)))
			plots <- c(plots, list(dae_l3_cache_misses_count(d)))
		}
	}

	plots <- c(plots, list(dae_branch_percent_mispredictions(d)))
	plots <- c(plots, list(dae_branch_mispredictions(d)))
	plots <- c(plots, list(dae_running_time(d)))
	plots <- c(plots, list(dae_instructions_completed(d)))
	plots <- c(plots, list(dae_tlb_data_misses(d)))
	plots <- c(plots, list(dae_percentage_cycles_stalled(d)))
	plots <- c(plots, list(dae_fraction_completed_to_issued(d)))

	require("gridExtra")

	grid_arrange_shared_legend(plots)

	#multiplot(plots, cols=2, mylegend)
}

grid_arrange_shared_legend <- function(plots) {
	g <- ggplotGrob(plots[[1]] + theme(legend.position="bottom"))$grobs
	legend <- g[[which(sapply(g, function(x) x$name) == "guide-box")]]
	lheight <- sum(legend$height)

	args <- lapply(plots, function(x) x + theme(legend.position="none"))
	args <- c(args, ncol = 2)

	grid.arrange(
				 do.call(arrangeGrob, args),
				 legend,
				 ncol = 1,
				 heights = unit.c(unit(1, "npc") - lheight, lheight))
}

dae_save_plot <- function(d, filename) {
	png(filename=filename, width=1920, height=1080)
	dae_plot(d)
	dev.off()
}

dae_plot_start_full <- function(filename) {
	tikz(file=filename, width=6.25432406254)
}

dae_plot_start_column <- function(filename) {
	tikz(file=filename, width=3.05797703058)
}

dae_plot_stop <- function(filename) {
	dev.off()
}


dae_frigo_summarize <- function(d) {
	new <- d[, lapply(.SD, mean), by = list(RecStop, Testsize)]

	return(new)
}

dae_frigo_running_time <- function(d) {
	p <- ggplot(d, aes(x=RecStop, y=((1e9*Seconds+Nanoseconds)/1e9)))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="Stop Criteria", y="Seconds", title="Running Time")
	p <- p + scale_x_continuous(breaks=unique(d$RecStop))
	p <- p + theme(legend.position="top", text = element_text(size=8))

	return(p)
}

dae_mat_running_time <- function(d) {
	require(scales)
	p <- ggplot(d, aes(x=log2(Testsize), y=((1e9*Seconds+Nanoseconds)/1e9), colour=Testfile, linetype=Testfile, shape=Testfile))
	p <- p + geom_line(size=LINESIZE)
	p <- p + geom_point(size=POINTSIZE)
	p <- p + labs(x="log(N)", y="Runingtime (s)")
	p <- p + scale_y_log10(breaks=c(0.0001, 0.001, 0.01, 0.1, 1, 10, 100), labels=trans_format("log10", math_format(10^.x)))
	p <- p + scale_x_continuous(breaks=(seq(floor(min(log2(d$Testsize))), ceiling(max(log2(d$Testsize))))))
	p <- p + theme(legend.position="top", text = element_text(size=8))

	return(p)
}

dae_plot_eps <- function(filename, func, d) {
	postscript(file=filename, width=6.25432406254, height=8.27452608275, family="Palatino", horizontal=F)
	func(d)
}

dae_plot_eps_full <- function(filename, func, d) {
	postscript(file=filename, width=6.25432406254, height=3, family="Palatino", horizontal=F)
	p <- func(d)
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + ggtitle(NULL)
	p <- p + theme(legend.title=element_blank())
	print(p)
	dev.off()
}

dae_plot_eps_column <- function(filename, func, d) {
	postscript(file=filename, width=3.05797703058, height=2, family="Palatino", horizontal=F)
	p <- func(d)
	p <- p + theme(legend.position="top", text = element_text(size=8))
	p <- p + ggtitle(NULL)
	p <- p + theme(legend.title=element_blank())
	print(p)
	dev.off()
}
