#define _DEFAULT_SOURCE
#define _GNU_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <inttypes.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <errno.h>
#include <sys/stat.h>
#include <measure.h>
#include <sys/types.h>
#include <dirent.h>

#include "main.h"
#include "xutil.h"
#include "inorder.h"
#include "bfs.h"
#include "dfs.h"
#include "veb.h"
#include "qsort.h"

void printusage(char *argv[]) {
	fprintf(stderr, "Usage: %s \n"
			"\t[-N #samples]\n"
			, argv[0]);
}

uint32_t read_input_file(char *filename, uint32_t **array) {
	int fd;
	uint32_t N, size;
	size = xfilesize(filename);
	N = size/sizeof(uint32_t);

	// Allocate memory to hold integers
	*array = xmalloc(size);

	// Open file containing integers and store file descriptor
	fd = open(filename, O_RDONLY);

	// Error check
	if (fd == -1) {
		xerror(strerror(errno), __LINE__, __FILE__);
	}

	// Read entire file and error check 
	if ( read(fd, *array, size) == -1 ) {
		xerror(strerror(errno), __LINE__, __FILE__);
	}

	// Close file descriptor
	close(fd);

	return N;
}

int main(int argc, char *argv[]) {
	char buf[256];
	char **files; // For storing filenames for testing.
	int opt;

	DIR *dir;
	struct dirent *entry;

	uint32_t files_count;
	uint32_t N = 0;
	uint32_t i, j, k, l, m, treesize,
			 REPS = 5,
			 OUTER_REPS = 4,
			 REPS_MEASURE = 10;

	uint32_t *storage = NULL;
	inorder_t *inorder;
	bfs_t *bfs;
	dfs_t *dfs;
	veb_t *veb;

	(void)bfs;
	(void)dfs;
	(void)veb;

	while ((opt = getopt(argc, argv, "")) != -1) {
		switch (opt) {
		default:
			printusage(argv);
			exit(EXIT_FAILURE);
		}
	}

	if (!measure_init()) {
		xerror("Error initializing measure library", __LINE__, __FILE__);
	}

	files_count = xfilesdir("DATA");
	files = xcalloc(files_count, sizeof(char *));

	if ((dir = opendir("DATA")) == NULL) {
		xerror(strerror(errno), __LINE__, __FILE__);
	}

	i = 0;
	while ( (entry = readdir(dir)) != NULL ) {
		if (entry->d_type == DT_REG) { /* If the entry is a regular file */
			files[i] = xmalloc(sizeof(char)*(256+5));
			memcpy(files[i], "DATA/", 5);
			memcpy(&files[i][5], entry->d_name, 256);
			i++;	
		}
	}

	/********* Inorder Tree Testing *********/

	uint32_t fractions = 8;
	struct inorder_search *in_search[REPS_MEASURE];

	srand48(43235);
	for (m = 0; m < OUTER_REPS; m++) {
		for (i = 0; i < REPS_MEASURE; i++) {
			in_search[i] = malloc(sizeof(struct inorder_search));
		}

		for (i = 0; i < files_count; i++) {
			N = read_input_file(files[i], &storage);

			for (j = 1; j <= fractions; j++) {
				treesize = N/2 * (1.0 + (double)j/fractions);

				inorder = inorder_create(storage, treesize, 0.5);

				for (l = 0; l < REPS_MEASURE; l++) {
					in_search[l]->inorder = inorder;
				}

				for (k = 0; k < REPS; k++) {
					snprintf(buf, 256, "%d", treesize);

					for (l = 0; l < REPS_MEASURE; l++) {
						in_search[l]->x = (uint32_t)mrand48();
					}

					measure("Inorder", buf, (testfunc)inorder_pred, (void*)&in_search, REPS_MEASURE);
				}

				free(inorder->storage);
				free(inorder);
			}

			free(storage);
		}

		for (i = 0; i < REPS_MEASURE; i++) {
			free(in_search[i]);
		}

		/********* BFS Tree Testing *********/

		struct bfs_search *bfs_str[REPS_MEASURE];

		for (i = 0; i < REPS_MEASURE; i++) {
			bfs_str[i] = malloc(sizeof(struct bfs_search));
		}

		srand48(43235);
		for (i = 0; i < files_count; i++) {
			N = read_input_file(files[i], &storage);
			quicksort(storage, 0, N-1);

			for (j = 1; j <= fractions; j++) {
				treesize = N/2 * (1.0 + (double)j/fractions);

				bfs = bfs_create(storage, treesize, 0.5);

				for (l = 0; l < REPS_MEASURE; l++) {
					bfs_str[l]->bfs = bfs;
				}

				for (k = 0; k < REPS; k++) {
					snprintf(buf, 256, "%d", treesize);

					for (l = 0; l < REPS_MEASURE; l++) {
						bfs_str[l]->x = (uint32_t)mrand48();
					}
					measure("BFS", buf, (testfunc)bfs_pred, (void*)&bfs_str, REPS_MEASURE);
				}

				free(bfs->storage);
				free(bfs);
			}

			free(storage);
		}

		for (i = 0; i < REPS_MEASURE; i++) {
			free(bfs_str[i]);
		}

		/********* DFS Tree Testing *********/
		
		struct dfs_search *dfs_str[REPS_MEASURE];

		for (i = 0; i < REPS_MEASURE; i++) {
			dfs_str[i] = malloc(sizeof(struct dfs_search));
		}

		srand48(43235);
		for (i = 0; i < files_count; i++) {
			N = read_input_file(files[i], &storage);
			quicksort(storage, 0, N-1);

			for (j = 1; j <= fractions; j++) {
				treesize = N/2 * (1.0 + (double)j/fractions);

				dfs = dfs_create(storage, treesize, 0.5);

				for (l = 0; l < REPS_MEASURE; l++) {
					dfs_str[l]->dfs = dfs;
				}

				for (k = 0; k < REPS; k++) {
					snprintf(buf, 256, "%d", treesize);

					for (l = 0; l < REPS_MEASURE; l++) {
						dfs_str[l]->x = (uint32_t)mrand48();
					}

					measure("DFS", buf, (testfunc)dfs_pred, (void*)&dfs_str, REPS_MEASURE);
				}

				free(dfs->storage);
				free(dfs);
			}

			free(storage);
		}

		for (i = 0; i < REPS_MEASURE; i++) {
			free(dfs_str[i]);
		}

		/********* vEB Tree Testing *********/

		struct veb_search *veb_str[REPS_MEASURE];

		for (i = 0; i < REPS_MEASURE; i++) {
			veb_str[i] = malloc(sizeof(struct veb_search));
		}

		srand48(43235);
		for (i = 0; i < files_count; i++) {
			N = read_input_file(files[i], &storage);
			quicksort(storage, 0, N-1);

			for (j = 1; j <= fractions; j++) {
				treesize = N/2 * (1.0 + (double)j/fractions);

				veb = veb_create(storage, treesize);

				for (l = 0; l < REPS_MEASURE; l++) {
					veb_str[l]->veb = veb;
				}

				for (k = 0; k < REPS; k++) {
					snprintf(buf, 256, "%d", treesize);

					for (l = 0; l < REPS_MEASURE; l++) {
						veb_str[l]->x = (uint32_t)mrand48();
					}

					measure("vEB", buf, (testfunc)veb_pred, (void*)&veb_str, REPS_MEASURE);
				}

				free(veb->table);
				free(veb->storage);
				free(veb);

			}

			free(storage);
		}

		for (i = 0; i < REPS_MEASURE; i++) {
			free(veb_str[i]);
		}
	}


	// Exit program successfully
	return EXIT_SUCCESS;
}
