#include <string.h>
#include <stdio.h>
#include <stdbool.h>
#include <inttypes.h>
#include <criterion/criterion.h>

#include "node.h"
#include "fibonacciheap.h"
#include "draw_fibonacci.h"

FibonacciHeap_t *heap;

void setup(void) {
	heap = MakeFibonacciHeap();
}

void teardown(void) {
	FreeFibonacciHeap(heap);
}

TestSuite(fibonacci_heap, .init=setup, .fini=teardown);

Test(fibonacci_heap, empty_heap) {
	cr_expect(NULL == FibonacciHeapDeleteMin(heap), "Testing DeleteMin returns 0 if heap is empty");
	cr_expect(NULL == FibonacciHeapFindMin(heap), "Testing FindMin returns 0 if heap is empty");
}

Test(fibonacci_heap, insert_decrease_delete) {
	Node_t *val;
	Node_t *node1, *node2, *node3, *node4, *node5, *node6, *node7, *node8;

	node1 = FibonacciHeapInsert(heap, -239);
	node2 = FibonacciHeapInsert(heap, 594);
	node3 = FibonacciHeapInsert(heap, 592);
	node4 = FibonacciHeapInsert(heap, 592);
	node5 = FibonacciHeapInsert(heap, -9746);
	node6 = FibonacciHeapInsert(heap, 0);
	node7 = FibonacciHeapInsert(heap, 1);
	node8 = FibonacciHeapInsert(heap, -1);

	(void)node1;
	FibonacciHeapDecreaseKey(heap, 10, node2);
	FibonacciHeapDecreaseKey(heap, 1000, node3);
	(void)node4;
	(void)node5;
	(void)node6;
	(void)node7;
	(void)node8;

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(-9746 == val->key, "Expected key: -9746 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(-408 == val->key, "Expected key: -408 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(-239 == val->key, "Expected key: -239 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(-1 == val->key, "Expected key: -1 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(0 == val->key, "Expected key: 0 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(1 == val->key, "Expected key: 1 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(584 == val->key, "Expected key: 584 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(592 == val->key, "Expected key: 592 got %"PRId64, val->key);
	free(val);
}

Test(fibonacci_heap, cascading_cuts_invoked_by_decrease_key) {
	Node_t *node2, *node5, *node12, *node13, *node16, *node17, *node19, *node24;
	Node_t *val;

	// Add 34 nodes
	FibonacciHeapInsert(heap, 1);
	node2 = FibonacciHeapInsert(heap, 2);
	FibonacciHeapInsert(heap, 3);
	FibonacciHeapInsert(heap, 4);
	node5 = FibonacciHeapInsert(heap, 5);
	FibonacciHeapInsert(heap, 6);
	FibonacciHeapInsert(heap, 7);
	FibonacciHeapInsert(heap, 8);
	FibonacciHeapInsert(heap, 9);
	FibonacciHeapInsert(heap, 10);
	FibonacciHeapInsert(heap, 11);
	node12 = FibonacciHeapInsert(heap, 12);
	node13 = FibonacciHeapInsert(heap, 13);
	FibonacciHeapInsert(heap, 14);
	FibonacciHeapInsert(heap, 15);
	node16 = FibonacciHeapInsert(heap, 16);
	node17 = FibonacciHeapInsert(heap, 17);
	FibonacciHeapInsert(heap, 18);
	node19 = FibonacciHeapInsert(heap, 19);
	FibonacciHeapInsert(heap, 20);
	FibonacciHeapInsert(heap, 21);
	FibonacciHeapInsert(heap, 22);
	FibonacciHeapInsert(heap, 23);
	node24 = FibonacciHeapInsert(heap, 24);
	FibonacciHeapInsert(heap, 25);
	FibonacciHeapInsert(heap, 26);
	FibonacciHeapInsert(heap, 27);
	FibonacciHeapInsert(heap, 28);
	FibonacciHeapInsert(heap, 29);
	FibonacciHeapInsert(heap, 30);
	FibonacciHeapInsert(heap, 31);
	FibonacciHeapInsert(heap, 32);
	FibonacciHeapInsert(heap, 33);
	FibonacciHeapInsert(heap, 34);

	// Make node2 the new min with key 0
	FibonacciHeapDecreaseKey(heap, 2, node2);

	// Delete node 24
	FibonacciHeapDelete(heap, node24);

	// Delete minimum node, resulting in linking
	val = FibonacciHeapDeleteMin(heap);
	cr_expect(val->key == 0, "Expected key: 0 got %"PRId64, val->key);
	free(val);

	DrawFibonacciHeap(heap, "rank5");

	// Expect one tree of size 5
	cr_expect(heap->min->rank == 5, "Expected rank: 5 got %"PRId64, heap->min->rank);

	// Delete nodes in lower part of tree, enabling potential cascading cuts
	FibonacciHeapDelete(heap, node12);
	DrawFibonacciHeap(heap, "marked1");
	FibonacciHeapDecreaseKey(heap, 3, node5);
	DrawFibonacciHeap(heap, "marked2");
	FibonacciHeapDelete(heap, node17);
	DrawFibonacciHeap(heap, "marked3");
	FibonacciHeapDecreaseKey(heap, 2, node16);

	DrawFibonacciHeap(heap, "cascading_cut_decrease_key");

	cr_expect(heap->min->rank == 4, "Expected rank: 4 got %"PRId64, heap->min->rank);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(1 == val->key, "Expected key: 1 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(2 == val->key, "Expected key: 2 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(3 == val->key, "Expected key: 3 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(4 == val->key, "Expected key: 4 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(6 == val->key, "Expected key: 6 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(7 == val->key, "Expected key: 7 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(8 == val->key, "Expected key: 8 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(9 == val->key, "Expected key: 9 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(10 == val->key, "Expected key: 10 got %"PRId64, val->key);
	free(val);

	FibonacciHeapInsert(heap, 12);

	FibonacciHeapDecreaseKey(heap, 2, node13);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(11 == val->key, "Expected key: 11 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(11 == val->key, "Expected key: 11 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(12 == val->key, "Expected key: 12 got %"PRId64, val->key);
	free(val);

	FibonacciHeapDelete(heap, node19);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(14 == val->key, "Expected key: 14 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(14 == val->key, "Expected key: 14 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(15 == val->key, "Expected key: 15 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(18 == val->key, "Expected key: 18 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(20 == val->key, "Expected key: 20 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(21 == val->key, "Expected key: 21 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(22 == val->key, "Expected key: 22 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(23 == val->key, "Expected key: 23 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(25 == val->key, "Expected key: 25 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(26 == val->key, "Expected key: 26 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(27 == val->key, "Expected key: 27 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(28 == val->key, "Expected key: 28 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(29 == val->key, "Expected key: 29 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(30 == val->key, "Expected key: 30 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(31 == val->key, "Expected key: 31 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(32 == val->key, "Expected key: 32 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(33 == val->key, "Expected key: 33 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(34 == val->key, "Expected key: 34 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(NULL == val, "Expected NULL");
	free(val);
}

Test(fibonacci_heap, cascading_cut_invoked_by_delete) {
	Node_t *node2, *node5, *node12, *node13, *node16, *node17, *node19, *node24;
	Node_t *val;

	// Add 34 nodes
	FibonacciHeapInsert(heap, 1);
	node2 = FibonacciHeapInsert(heap, 2);
	FibonacciHeapInsert(heap, 3);
	FibonacciHeapInsert(heap, 4);
	node5 = FibonacciHeapInsert(heap, 5);
	FibonacciHeapInsert(heap, 6);
	FibonacciHeapInsert(heap, 7);
	FibonacciHeapInsert(heap, 8);
	FibonacciHeapInsert(heap, 9);
	FibonacciHeapInsert(heap, 10);
	FibonacciHeapInsert(heap, 11);
	node12 = FibonacciHeapInsert(heap, 12);
	node13 = FibonacciHeapInsert(heap, 13);
	FibonacciHeapInsert(heap, 14);
	FibonacciHeapInsert(heap, 15);
	node16 = FibonacciHeapInsert(heap, 16);
	node17 = FibonacciHeapInsert(heap, 17);
	FibonacciHeapInsert(heap, 18);
	node19 = FibonacciHeapInsert(heap, 19);
	FibonacciHeapInsert(heap, 20);
	FibonacciHeapInsert(heap, 21);
	FibonacciHeapInsert(heap, 22);
	FibonacciHeapInsert(heap, 23);
	node24 = FibonacciHeapInsert(heap, 24);
	FibonacciHeapInsert(heap, 25);
	FibonacciHeapInsert(heap, 26);
	FibonacciHeapInsert(heap, 27);
	FibonacciHeapInsert(heap, 28);
	FibonacciHeapInsert(heap, 29);
	FibonacciHeapInsert(heap, 30);
	FibonacciHeapInsert(heap, 31);
	FibonacciHeapInsert(heap, 32);
	FibonacciHeapInsert(heap, 33);
	FibonacciHeapInsert(heap, 34);

	// Make node2 the new min with key 0
	FibonacciHeapDecreaseKey(heap, 2, node2);

	// Delete node 24
	FibonacciHeapDelete(heap, node24);

	// Delete minimum node, resulting in linking
	val = FibonacciHeapDeleteMin(heap);
	cr_expect(val->key == 0, "Expected key: 0 got %"PRId64, val->key);
	free(val);

	// Expect one tree of size 5
	cr_expect(heap->min->rank == 5, "Expected rank: 5 got %"PRId64, heap->min->rank);

	// Delete nodes in lower part of tree, enabling potential cascading cuts
	FibonacciHeapDecreaseKey(heap, 3, node5);
	FibonacciHeapDelete(heap, node12);
	FibonacciHeapDecreaseKey(heap, 2, node16);
	FibonacciHeapDelete(heap, node17);

	DrawFibonacciHeap(heap, "casading_cut_delete");

	cr_expect(heap->min->rank == 4, "Expected rank: 4 got %"PRId64, heap->min->rank);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(1 == val->key, "Expected key: 1 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(2 == val->key, "Expected key: 2 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(3 == val->key, "Expected key: 3 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(4 == val->key, "Expected key: 4 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(6 == val->key, "Expected key: 6 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(7 == val->key, "Expected key: 7 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(8 == val->key, "Expected key: 8 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(9 == val->key, "Expected key: 9 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(10 == val->key, "Expected key: 10 got %"PRId64, val->key);
	free(val);

	FibonacciHeapInsert(heap, 12);

	FibonacciHeapDecreaseKey(heap, 2, node13);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(11 == val->key, "Expected key: 11 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(11 == val->key, "Expected key: 11 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(12 == val->key, "Expected key: 12 got %"PRId64, val->key);
	free(val);

	FibonacciHeapDelete(heap, node19);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(14 == val->key, "Expected key: 14 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(14 == val->key, "Expected key: 14 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(15 == val->key, "Expected key: 15 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(18 == val->key, "Expected key: 18 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(20 == val->key, "Expected key: 20 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(21 == val->key, "Expected key: 21 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(22 == val->key, "Expected key: 22 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(23 == val->key, "Expected key: 23 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(25 == val->key, "Expected key: 25 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(26 == val->key, "Expected key: 26 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(27 == val->key, "Expected key: 27 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(28 == val->key, "Expected key: 28 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(29 == val->key, "Expected key: 29 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(30 == val->key, "Expected key: 30 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(31 == val->key, "Expected key: 31 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(32 == val->key, "Expected key: 32 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(33 == val->key, "Expected key: 33 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(34 == val->key, "Expected key: 34 got %"PRId64, val->key);
	free(val);

	val = FibonacciHeapDeleteMin(heap);
	cr_expect(NULL == val, "Expected NULL");
	free(val);
}

Test(fibonacci_heap, max_height) {
	int i = 0;
	int w = 99;
	int n = 99;
	int height = 0;
	Node_t *node;
	Node_t *nodes[n];

	nodes[i++] = FibonacciHeapInsert(heap, w--);
	nodes[i++] = FibonacciHeapInsert(heap, w--);
	nodes[i++] = FibonacciHeapInsert(heap, w--);

	free(FibonacciHeapDeleteMin(heap));

	while (i < n) {
		nodes[i++] = FibonacciHeapInsert(heap, w--);
		nodes[i++] = FibonacciHeapInsert(heap, w--);
		nodes[i++] = FibonacciHeapInsert(heap, w--);

		free(FibonacciHeapDeleteMin(heap));

		FibonacciHeapDelete(heap, nodes[i-3]);
	}

	node = FibonacciHeapFindMin(heap);
	while (	NULL != node->child ) {
		node = node->child;
		height++;
	}

	cr_expect(heap->n-1 == (uint64_t)height, "Expected key: %"PRId64" got %d", heap->n-1, height);

	DrawFibonacciHeap(heap, "max_height");
}

bool ispowerof2(unsigned int x) {
	return x && !(x & (x - 1));
}

Test(fibonacci_heap, max_rank) {
	int RANK = 4;
	int j;
	int k = 0;
	int i = 0;
	int n = 0;
	int w = 0;
	int m = (1 << RANK) + RANK - 1;
	int po2 = 4;
	Node_t *nodes[m];

	while ( i < m ) {
		nodes[i++] = FibonacciHeapInsert(heap, INT64_MIN);
		n++;

		// Insert nodes two form new tree of the latest rank
		while ( k < po2 ) {
			nodes[i++] = FibonacciHeapInsert(heap, w++);
			n++;
			k++;
		}

		DrawFibonacciHeap(heap, "max_rank_next_before");

		free(FibonacciHeapDeleteMin(heap));
		n--;

		DrawFibonacciHeap(heap, "max_rank_next");

		// Delete nodes not needed
		j = i-1;
		while ( !ispowerof2(n-1) ) {
			FibonacciHeapDelete(heap, nodes[j]);
			n--;
			j -= 2;
		}

		po2 *= 2;
	}

	cr_expect(heap->min->rank == (uint64_t)RANK, "Expected key: %d got %"PRId64, RANK, heap->min->rank);

	DrawFibonacciHeap(heap, "max_rank");
}

//TODO: Test decrease key integer overflow
