#define _DEFAULT_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <errno.h>
#include <inttypes.h>

#include "xutil.h"
#include "node.h"
#include "binaryheap.h"

BinaryHeap_t *
MakeBinaryHeap() {
	BinaryHeap_t *heap = xmalloc( sizeof(BinaryHeap_t) );

	heap->A = NULL;
	heap->size = 0;
	heap->n = 0;

	return heap;
}

void
FreeBinaryHeap(BinaryHeap_t *h) {
	uint64_t i;
	if (NULL != h) {
		if (NULL != h->A) {
			for (i = 0; i < h->n; i++) {
				if ( NULL != h->A[i] ) {
					free(h->A[i]);
				}
			}
			free(h->A);
		}
		free(h);
	}
}

static inline void 
decreaseKeyBinary(BinaryHeap_t *h, Node_t *x) {
	Node_t **A = h->A;
	uint64_t j;

	if ( unlikely(A[0] == x) ) {
		return;
	}

	do {
		j = (x->rank-1)/2;
#ifdef COUNT_COMPARISONS
		bin_comps++;
		bin_decreasekey_comps++;
#endif
		if (x->key < A[j]->key) {
			A[x->rank] = A[j];
			A[j]->rank = x->rank;
			A[j] = x;
			x->rank = j;
		} else {
			return;
		}
	} while (j > 0);
}

static inline void
swopBinaryHeap(Node_t **A, uint64_t n, uint64_t i, uint64_t j) {
	Node_t *in, *temp, *temp1;

	in = A[n];
	while ( j < n ) {
		temp   = A[j];
		temp1  = A[j+1];

#ifdef COUNT_COMPARISONS
		bin_comps++;
		bin_deletemin_comps++;
#endif
		if ( temp1->key < temp->key ) {
			temp = temp1;
			j++;
		}

#ifdef COUNT_COMPARISONS
		bin_comps++;
		bin_deletemin_comps++;
#endif
		if ( temp->key < in->key ) {
			A[i] = temp;
			temp->rank = i;
			i = j;
			j = (i << 1) + 1;
			continue;
		}

		break;
	}
	A[i] = in;
	in->rank = i;
}

bool BinaryHeapIsEmpty(BinaryHeap_t *h) {
	return (bool) h->n == 0;
}

Node_t *
BinaryHeapFindMin(BinaryHeap_t *h) {
	if (h->n == 0) {
		return NULL;
	}
	return h->A[0];
}

Node_t *
BinaryHeapInsert(BinaryHeap_t *h, int64_t in) {
	Node_t **A;
	uint64_t i, j;
	uint64_t n = h->n;
	Node_t *node = MakeNode(in);

	// Realloc A to double size
	if ( unlikely(n + 1 > h->size) ) {
		h->size = (n + 1) * 2;
		h->A = xrealloc(h->A, h->size*sizeof(Node_t));
	}

	A = h->A;

	i = n;
	while ( i > 0 ) {
		j = (uint64_t)(i-1)/2;	
#ifdef COUNT_COMPARISONS
		bin_comps++;
		bin_insert_comps++;
#endif
		if ( node->key < A[j]->key ) {
			A[i] = A[j];
			A[j]->rank = i;
			i = j;
			continue;
		}
		break;
	}
	A[i] = node;
	node->rank = i;

	h->n++;

	return node;
}

Node_t *
BinaryHeapDeleteMin(BinaryHeap_t *h) {
	Node_t **A, *x;
	uint64_t n = h->n;

	A = h->A;

	if ( unlikely(n == 0) ) {
		return NULL;
	} else if ( unlikely(n == 1) ) {
		h->size = 0;
		h->n = 0;
		return A[0];
	}

	x = A[0];

	h->n = --n;                   // Invariant n >= 1
	swopBinaryHeap(A, n, 0, 1);

	if ( h->n < (h->size >> 2) ) {
		h->size = (h->size >> 1);
		h->A = xrealloc(h->A, h->size*sizeof(int64_t));
	}

	return x;
}

void
BinaryHeapDecreaseKeyTo(BinaryHeap_t *h, int64_t val, Node_t *x) {
#ifdef COUNT_COMPARISONS
	bin_decreasekey_comps++;
	bin_comps++;
#endif
	if ( val < x->key ) {
		x->key = val;
		decreaseKeyBinary(h, x);
	}
}

void
BinaryHeapDecreaseKey(BinaryHeap_t *h, uint64_t val, Node_t *x) {
	if ( unlikely( (x->key < 0 && (val >= INT64_MAX-1 || (uint64_t)(val - x->key) > INT64_MAX-1)) || 
				   (x->key >= 0 && val > (uint64_t)x->key && (uint64_t)(val - x->key) > INT64_MAX-1) ) ) {
		x->key = INT64_MIN;
	} else {
		x->key -= val;		
	}
	decreaseKeyBinary(h, x);
}

void
BinaryHeapDelete(BinaryHeap_t *h, Node_t *x) {
	Node_t **A;
	uint64_t idx, n = h->n;

	A = h->A;

	if ( unlikely(x == A[0]) ) {
		free(BinaryHeapDeleteMin(h));
		return;
	} 

	idx = x->rank;

	h->n = --n; // Invariant n >= 1
	swopBinaryHeap(A, n, idx, (idx << 1)+1);

	if ( h->n < (h->size >> 2) ) {
		h->size = (h->size >> 1);
		h->A = xrealloc(h->A, h->size*sizeof(int64_t));
	}

	free(x);
}
