#define _DEFAULT_SOURCE

#include "streams.h"
#include "stream_common.h"
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/types.h>

#include <assert.h>

#include <inttypes.h>

typedef struct input {
	int fd;
	int dir;
	uint64_t length;
	uint64_t cur;
	void *map;
	int map_size;
	uint64_t bp;
	uint64_t B;
	uint64_t bpOffset;
	uint64_t offset;
} input_stream;

typedef struct output {
	int fd;
	int dir;
	uint64_t length;
	uint64_t cur;
	void *map;
	int map_size;
	uint64_t bp;
	uint64_t B;
	uint64_t bpOffset;
	uint64_t offset;
} output_stream;

static input_stream **in_list;
static int in_len;

static output_stream **out_list;
static int out_len;

/*
 * INPUT STREAM OPERATIONS
 */

__attribute__ ((visibility("default")))
int mi_open(const char *f, uint64_t start, uint64_t length, int B, int dir) {
	int id = 0;
	long page_size;

	assert(B > 0);

	id = stream_find_struct_storage((uintptr_t **)&in_list, &in_len);

	input_stream *s =  xmalloc(sizeof(input_stream));
	in_list[id] = s;

	s->fd = open(f, O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR);

	if(s->fd < 0) {
		printf("Error in mi_open: '%s'\n", strerror(errno));
		exit(1);
	}

	s->cur = sizeof(uint32_t) * start;
	s->B = B;
	s->length = length;

	if (dir == STREAM_FORWARDS) {
		s->bp = 0;
	} else {
		if (s->B >= length) {
			s->cur -= length * sizeof(uint32_t);
			s->bp = length;
		} else {
			s->cur -= s->B*sizeof(uint32_t);
			s->bp = B;
		}
	}

	page_size = sysconf(_SC_PAGE_SIZE);
	s->bpOffset = s->cur % page_size;
	s->offset = s->cur - s->bpOffset;
	s->dir = dir;
	s->map_size = sizeof(uint32_t) * B + s->bpOffset;

	s->map = mmap(NULL, s->map_size, PROT_READ, 
			MAP_PRIVATE, s->fd, s->offset);

	if (s->map == MAP_FAILED) {
		printf("Unable to map@open: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	return id;
}

// This functions moves the map to begin at i->cur.
static inline void __mi_move_map(input_stream *i) {
	int res;
	long page_size;

	// Unmap earlier mapped memory.
	res = munmap(i->map, i->map_size);
	if (res == -1) {
		printf("Unable to unmap@next: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	if (i->dir == STREAM_BACKWARDS && i->B > i->length) {
		// So, cur is decremented by i->B before we enter this, if we have
		// less left to read than i->B, we need to add the difference
		// between i->B and i->length.
		i->cur += (i->B - i->length) * sizeof(uint32_t);
	}

	if (i->dir == STREAM_FORWARDS) {
		i->bp = 0;
	} else {
		// \equiv min(i->length, i->B);
		i->bp = (i->length < i->B) ? i->length : i->B;
	}

	// Mmap needs a size aligned with PAGE_SIZE.
	page_size = sysconf(_SC_PAGE_SIZE);

	// bpOffset is buffer offset into the nearest aligned page.
	i->bpOffset = i->cur % page_size;

	// offset is the beginning of the aligned page.
	i->offset = i->cur - i->bpOffset;

	// If our buffer is bigger than the remaining things we need to read.
	if (i->length < i->B) {
		// Just allocate a size that corresponds to the rest of the file.
		i->map_size = sizeof(uint32_t) * i->length + i->bpOffset;
	} else {
		// Else allocate the size of our buffer.
		i->map_size = sizeof(uint32_t) * i->B + i->bpOffset;
	}

	i->map = mmap(NULL, i->map_size, PROT_READ, MAP_PRIVATE, i->fd, i->offset);

	if (i->map == MAP_FAILED) {
		printf("Unable to map@next: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

#ifdef MADVISE
	if (madvise(i->map, i->map_size, MADV_SEQUENTIAL) == -1) {
		fprintf(stderr, "Unable to madvise@next: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}
#endif
}

__attribute__ ((visibility("default")))
uint32_t mi_next(int id) {
	input_stream *i = in_list[id];
	assert(i->dir == STREAM_FORWARDS);
	assert(i->length > 0);

	if (i->bp == i->B) {
		__mi_move_map(i);
	}

	i->cur += sizeof(uint32_t);
	i->length--;

	uint32_t *valindex = (uint32_t *)((char *)i->map
		+ i->bpOffset
		+ (i->bp * sizeof(uint32_t)));

	i->bp++;

	return *valindex;
}

__attribute__ ((visibility("default")))
uint32_t mi_last(int id) {
	input_stream *i = in_list[id];
	assert(i->dir == STREAM_BACKWARDS);
	assert(i->length > 0);

	if (i->bp == 0) {
		__mi_move_map(i);
	}

	i->bp--;
	i->length--;
	i->cur -= sizeof(uint32_t);

	uint32_t *valindex = (uint32_t *)((char *)i->map
		+ i->bpOffset
		+ (i->bp * sizeof(uint32_t)));

	return *valindex;
}

__attribute__ ((visibility("default")))
int mi_eos(int id) {
	input_stream *i = in_list[id];

	if (i->length == 0) {
		return 1;
	}

	return 0;
}

__attribute__ ((visibility("default")))
void mi_close(int id) {
	int res;
	input_stream *i = in_list[id];

	res = munmap(i->map, i->map_size);

	if (res == -1) {
		printf("Unable to unmap@close: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	close(i->fd);

	free(i);

	in_list[id] = 0;
}

__attribute__ ((visibility("default")))
void mi_seek(int id, long offset, int length) {
	input_stream *i = in_list[id];

	i->length = length;
	// TODO: Fix last decrement, it apparently doesn't work.
	if (i->dir == STREAM_BACKWARDS) {
		i->cur =  sizeof(uint32_t) * offset - sizeof(uint32_t) * i->B;
	} else {
		i->cur =  sizeof(uint32_t) * offset;
	}

	__mi_move_map(i);
}

/*
 * OUTPUT STREAM OPERATIONS
 */

__attribute__ ((visibility("default")))
int mo_create(const char *f, uint64_t start, uint64_t length, int B, int dir) {
	int od = 0;
	int r;
	long page_size;
	struct stat stat_buf;

	od = stream_find_struct_storage((uintptr_t **)&out_list, &out_len);

	output_stream *o = xmalloc(sizeof(output_stream));
	out_list[od] = o;

	o->fd = open(f, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);

	if(o->fd < 0) {
		printf("Error in mo_create: '%s'\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	r = fstat(o->fd, &stat_buf);

	if (r == -1) {
		printf("Unable to stat in mo_create: '%s'\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	//o->end = sizeof(uint32_t) * (start + length);
	o->length = length;
	o->cur = sizeof(uint32_t) * start;
	o->B = B;

	if (dir == STREAM_FORWARDS) {
		o->bp = 0;
	} else {
		if (o->B > length) {
			o->cur -= length*sizeof(uint32_t);
			o->bp = length;
		} else {
			o->cur -= o->B*sizeof(uint32_t);
			o->bp = B;
		}
	}

	// Expand the file so we can write to it.
	if ((uint64_t)stat_buf.st_size < o->cur + o->length * sizeof(uint32_t)) {
		r = ftruncate(o->fd, o->cur + o->length * sizeof(uint32_t));

		if (r == -1) {
			printf("Unable to truncate in mo_create: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}
	}

	page_size = sysconf(_SC_PAGE_SIZE);
	o->dir = dir;
	o->bpOffset = o->cur % page_size;
	o->offset = o->cur - o->bpOffset;
	o->map_size = sizeof(uint32_t) * B + o->bpOffset;

	o->map = mmap(NULL, o->map_size, PROT_WRITE, 
			MAP_SHARED, o->fd, o->offset);

	if (o->map == MAP_FAILED) {
		printf("Unable to map@create: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	return od;
}

void __mo_move_map(output_stream *o) {
	int res;
	long page_size;

	res = munmap(o->map, o->map_size);

	if (res == -1) {
		printf("Unable to unmap@write: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	if (o->dir == STREAM_BACKWARDS && o->B > o->length) {
		// So, cur is decremented by o->B before we enter this, if we have less
		// left to read than o->B, we need to add the difference between o->B
		// and o->length.
		o->cur += (o->B - o->length)*sizeof(uint32_t);
	}

	if (o->dir == STREAM_FORWARDS) {
		o->bp = 0;
	} else {
		// \equiv min(o->length, o->B);
		o->bp = (o->length < o->B) ? o->length : o->B;
	}

	page_size = sysconf(_SC_PAGE_SIZE);
	o->bpOffset = o->cur % page_size;
	o->offset = o->cur - o->bpOffset;

	// If our buffer is bigger than the remaining things we need to read.
	if (o->length < o->B) {
		// Just allocate a size that corresponds to the rest of the file.
		o->map_size = o->length * sizeof(uint32_t) + o->bpOffset;
	} else {
		// Else allocate the size of our buffer.
		o->map_size = sizeof(uint32_t) * o->B + o->bpOffset;
	}

	o->map = mmap(NULL, o->map_size, PROT_WRITE, MAP_SHARED, o->fd, 
			o->offset);

	if (o->map == MAP_FAILED) {
		fprintf(stderr, "Unable to mmap@write: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

#ifdef MADVISE
	if (madvise(o->map, o->map_size, MADV_SEQUENTIAL) == -1) {
		fprintf(stderr, "Unable to madvise@write: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}
#endif
}

__attribute__ ((visibility("default")))
void mo_write(int od, uint32_t v) {
	output_stream *o = out_list[od];
	assert(o->dir == STREAM_FORWARDS);

	if (o->bp == o->B) {
		__mo_move_map(o);
	}

	uint32_t *values = (uint32_t *)((char *)o->map + o->bpOffset);
	values[o->bp] = v;
	o->bp++;

	o->cur += sizeof(uint32_t);
}

__attribute__ ((visibility("default")))
void mo_prepend(int od, uint32_t v) {
	output_stream *o = out_list[od];
	assert(o->dir == STREAM_BACKWARDS);

	if (o->bp == 0) {
		__mo_move_map(o);
	}

	o->bp--;
	o->length--;
	o->cur -= sizeof(uint32_t);

	uint32_t *values = (uint32_t *)((char *)o->map + o->bpOffset);
	values[o->bp] = v;
}

__attribute__ ((visibility("default")))
void mo_close(int od) {
	output_stream *o = out_list[od];
	int res;

	res = munmap(o->map, o->map_size);

	if (res == -1) {
		printf("Unable to unmap@close: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	close(o->fd);

	free(o);

	out_list[od] = 0;
}

__attribute__ ((visibility("default")))
void mo_seek(int od, long offset, int length) {
	output_stream *o = out_list[od];

	o->length = length;
	o->cur = sizeof(uint32_t) * offset;

	__mo_move_map(o);
}
