#!/usr/bin/env bash
EXE=./TestStream
REPEAT_COUNT=20  # Times we repeat each test
TESTMAX=2140    # Maximal number of tests
RESULTS_FOLDER=BENCHMARK_RESULTS
TMP_FOLDER=tmp

DEBUG="0"

# TODO:
# - Check if buffer size is above write/read size?
# - Switch around repetitions, so they are outside.

REPETITION=0
TEST=0

MEMLIMIT=0
MEMLOAD=0

DISKLIMIT=0
DISKLOAD=0

# Don't allow memory usage over 1GiB
MEMLIMIT=$(( 1024 * 1024 * 1024 ))

# Don't allow disk usage over 4GiB
DISKLIMIT=$(( 1024*1024*4096 ))

# Element write range.
# 1MiB, 2MiB, 5MiB, 10MiB, 15MiB, 20MiB, 30MiB, 40MiB, 50MiB, 75MiB, 100MiB
declare -a Nrange=(262144 524288 1310720 2621440 3932160 5242880 7864320 10485760 13107200 19660800 26214400)

# The buffer size, per thread.
# 2KiB, 4KiB, 25KiB, 52Kib, 104KiB, 1MiB, 5MiB, 10MiB, 100MiB
declare -a Brange=(512 1024 6400 13312 26624 262144 1048576 1310720)

# Amount of writers/readers to use.
declare -a krange=(16)

# Run all 4 tests.
declare -a Srange=(3 4)

# Do both write/read test.
declare -a trange=('w' 'r')

trap "tput cvvis; echo \"\"; exit" SIGINT

function clearCaches() {
	if [[ ${UID} -eq 0 ]]; then
		echo 3 > /proc/sys/vm/drop_caches
		blockdev --flushbufs /dev/sda
		sync
	fi
}

function clearTmp() {
	t=$1

	# Delete files on run, if we are writing.
	if [[ ${t} == "w" ]]; then
		if [[ ${DEBUG} != "1" ]]; then
			rm -f ${TMP_FOLDER}/file*
		fi
	fi
}

function performTest() {
	local N=$1 # Output/input size in uint32_t
	local k=$2 # open files
	local S=$3 # Stream type
	local t=$4 # Test type (read/write)
	local B=$5 # Buffer size, in uint32_t

	# Clear the lines for output
	echo -en "\r"
	tput el
	tput cuu1
	echo -en "\r"
	tput el

	echo "Performing iteration ${REPETITION}/${REPEAT_COUNT} of test ${TEST}/${TESTMAX}"
	printf "t: %s  S: %d  k: %4d  N: %8d  B: %8d" $t $S $k $N $B

	# Where we save our data.
	OUTFILE=""

	# Specify the B parameter
	if [[ -n ${B} || ${B} != "" ]]; then
		OUTFILE="${RESULTS_FOLDER}/${S}.${t}.${k}.${N}.${B}.csv"
		B="-B ${B}"
	else
		B=""
		OUTFILE="${RESULTS_FOLDER}/${S}.${t}.${k}.${N}.csv"
	fi

	if [[ ${DEBUG} != "1" ]]; then
		${EXE} -D ${TMP_FOLDER} -S ${S} -N ${N} -t ${t} ${B} -k ${k} >> ${OUTFILE};
	else
		sleep 0.5
	fi
}

function runUnbuffered() {
	local N=$1
	local k=$2
	local S=$3

	for t in ${trange[@]}; do
		DISKLOAD=$(( ${k} * ${N} * 4 ))

		if [[ ${DISKLOAD} -lt ${DISKLIMIT} ]]; then
			TEST=$(( ${TEST} + 1 ))

			clearTmp ${t}
			performTest $N $k $S $t
			clearCaches

			# Delete files after we've read for the last time.
			if [[ ${t} == "r" ]]; then
				if [[ ${DEBUG} != "1" ]]; then
					rm -f ${TMP_FOLDER}/file*
				fi
			fi
		fi
	done
}

function runBuffered() {
	local N=$1
	local k=$2
	local S=$3

	for B in ${Brange[@]}; do
		for t in ${trange[@]}; do
			MEMLOAD=$(( ${k} * ${B} * 4 ))	
			DISKLOAD=$(( ${k} * ${N} * 4 ))

			if [[ ${MEMLOAD} -lt ${MEMLIMIT} &&  ${DISKLOAD} -lt ${DISKLIMIT} ]]; then
				TEST=$(( ${TEST} + 1 ))


				clearTmp ${t}
				performTest $N $k $S $t $B
				clearCaches

				# Delete files after we've read for the last time.
				if [[ ${t} == "r" ]]; then
					if [[ ${DEBUG} != "1" ]]; then
						rm -f ${TMP_FOLDER}/file*
					fi
				fi
			fi
		done
	done
}

#LEN=$(( ${#Nrange[@]} * ${#Brange[@]} * ${#krange[@]} * ${#Srange[@]} * ${#trange[@]} ))
#echo $LEN

tput civis
mkdir -p ${RESULTS_FOLDER}

echo ""

REPETITION=0
for r in `seq 1 ${REPEAT_COUNT}`; do
	REPETITION=$(( ${REPETITION} + 1 ))
	TEST=0

	for N in ${Nrange[@]}; do
		for k in ${krange[@]}; do
			for S in ${Srange[@]}; do

				if [[ ${S} -eq 1 || ${S} -eq 2 ]]; then
					runUnbuffered ${N} ${k} ${S}
				else
					runBuffered ${N} ${k} ${S}
				fi

			done
		done
	done
done

tput cvvis
echo ""
