#define _GNU_SOURCE

#include "streams.h"
#include "stream_common.h"
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdbool.h>

#include <assert.h>

#ifdef REUSEFDS
typedef struct {
	char *file;
	int fd;
	int refcount;
} fdinfo;
#endif

typedef struct input {
#ifdef REUSEFDS
	fdinfo *fdentry;	
	off_t file_offset;
#endif
	int fd;
	uint64_t end;
	uint64_t cur;
	uint32_t *restrict buf;
	uint64_t bp;
	unsigned int B;
	bool free_buffer;
	int dir;  // Direction to read.
} input_stream;

typedef struct output {
#ifdef REUSEFDS
	fdinfo *fdentry;	
	off_t file_offset;
#endif
	int fd;
	uint32_t *restrict buf;
	uint64_t bp;
	unsigned int B;
	int dir;  // Direction to read.
	bool free_buffer;
} output_stream;

#ifdef REUSEFDS
static map_t map_in;
static map_t map_out;
#endif

static input_stream **in_list;
static int in_len;

static output_stream **out_list;
static int out_len;

__attribute__ ((visibility("default")))
int bi_open(const char *restrict f, uint64_t start, uint64_t length, int B, int dir) {
	int id = 0;

	id = stream_find_struct_storage((uintptr_t **)&in_list, &in_len);

	input_stream *restrict s =  xmalloc(sizeof(input_stream));
	in_list[id] = s;

	// Allocate the buffer on first use.
	s->buf = NULL;
	s->B = B;
	s->dir = dir;

	if (dir == STREAM_FORWARDS) {
		s->bp = B;
	} else {
		s->bp = 0;
	}


#ifdef REUSEFDS
	fdinfo *fdentry;

	// Create the map_in if it doesn't exist.
	if (map_in == NULL) {
		map_in = hashmap_new();
		if (map_in == NULL) {
			fprintf(stderr, "Error: Initialized map_in is null\n");
			exit(EXIT_FAILURE);
		}
	}

	// Find a file descriptor if we have one.
	int r = hashmap_get(map_in, (char *)f, (void **)&fdentry);

	if (r == MAP_MISSING) { // Element not found
		fdentry = xmalloc(sizeof(fdinfo));
		fdentry->fd       = open(f, O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR);
		fdentry->refcount = 1;	
		fdentry->file     = strdup(f); // We need to store this.

		if(fdentry->fd < 0) {
			printf("Error in bi_open: '%s'\n", strerror(errno));
			exit(1);
		}

		r = hashmap_put(map_in, fdentry->file, fdentry);
		if (r == MAP_OMEM) {
			fprintf(stderr, "Out of memory adding to map_in\n");
			exit(EXIT_FAILURE);
		}
	} else {
		fdentry->refcount++;
	}

	s->fd          = fdentry->fd;
	s->fdentry     = fdentry;
#else
	s->fd = open(f, O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR);

	if(s->fd < 0) {
		printf("Error in bi_open: '%s'\n", strerror(errno));
		exit(1);
	}
#endif

	s->free_buffer = true;
	if (dir == STREAM_FORWARDS) {
		s->cur = sizeof(uint32_t) * start;
		s->end = sizeof(uint32_t) * (start + length);
	} else {
		s->cur = sizeof(uint32_t) * start;
		s->end = sizeof(uint32_t) * (start - length);
	}

	if (lseek(s->fd, s->cur, SEEK_SET) == -1) {
		fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
				__LINE__, __FILE__, strerror(errno));
		exit(EXIT_FAILURE);
	}

#ifdef REUSEFDS
	s->file_offset = s->cur;
#endif

	return id;
}

__attribute__ ((visibility("default")))
void bi_setbuffer(int id, int B, uint32_t *restrict buf) {
	input_stream *restrict i = in_list[id];

	assert(i->buf == NULL);

	if (i->dir == STREAM_FORWARDS) {
		i->bp = B;
	} else {
		i->bp = 0;
	}

	i->B = B;
	i->buf = buf;
	i->free_buffer = false;
}

__attribute__ ((visibility("default")))
uint32_t bi_next(int id) {
	uint64_t r;
	int err;

	input_stream *restrict i = in_list[id];
	assert(i->dir == STREAM_FORWARDS);

	// Allocate the buffer on first use.
	if (i->buf == NULL) {
		i->buf =  xcalloc(sizeof(uint32_t), i->B);
	}

	if (i->bp == i->B) {
		if (i->end - i->cur < i->B * sizeof(uint32_t)) {
			r = i->end - i->cur;
		} else {
			r = sizeof(uint32_t) * i->B;
		}

#ifdef REUSEFDS
		if (i->file_offset != lseek(i->fd, 0, SEEK_CUR)) {
			if (lseek(i->fd, i->file_offset, SEEK_SET) == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
			}
		}
#endif

		err = read(i->fd, i->buf, r);
		if (err == -1) {
			errorAndDie(5);
		}

#ifdef REUSEFDS
		i->file_offset = lseek(i->fd, 0, SEEK_CUR);
		if (i->file_offset == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
		}
#endif

		i->bp = 0;
	}

	i->cur += sizeof(uint32_t);

	return i->buf[i->bp++];
}

__attribute__ ((visibility("default")))
uint32_t bi_last(int id) {
	int64_t r;
	int err;

	input_stream *restrict i = in_list[id];
	assert(i->dir == STREAM_BACKWARDS);

	// Allocate the buffer on first use.
	if (i->buf == NULL) {
		i->buf =  xcalloc(sizeof(uint32_t), i->B);
	}

	if (i->bp == 0) {
		if (i->cur - i->end < i->B * sizeof(uint32_t)) {
			r = i->cur - i->end;
		} else {
			r = sizeof(uint32_t) * i->B;
		}

#ifdef REUSEFDS
		if (i->file_offset != lseek(i->fd, 0, SEEK_CUR)) {
			if (lseek(i->fd, i->file_offset, SEEK_SET) == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
			}
		}
#endif

		if (lseek(i->fd, -r, SEEK_CUR) == -1) {
			printf("Error in bi_last1: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		err = read(i->fd, i->buf, r);
		if (err == -1) {
			errorAndDie(10);
		}

		if (lseek(i->fd, -r, SEEK_CUR) == -1) {
			printf("Error in bi_last2: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

#ifdef REUSEFDS
		i->file_offset = lseek(i->fd, 0, SEEK_CUR);
		if (i->file_offset == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
		}
#endif

		i->bp = r/4;
	}

	i->cur -= sizeof(uint32_t);

	return i->buf[--i->bp];
}

__attribute__ ((visibility("default")))
int bi_eos(int id) {
	input_stream *restrict i = in_list[id];

	if ((i->dir == STREAM_FORWARDS && i->cur >= i->end) || 
			(i->dir == STREAM_BACKWARDS && i->cur <= i->end)) {
		return 1;
	}

	return 0;
}

__attribute__ ((visibility("default")))
void bi_close(int id) {
	input_stream *restrict i = in_list[id];

#ifdef REUSEFDS
	if (i->fdentry->refcount == 1) {
		fdinfo *fdentry = i->fdentry;
		hashmap_remove(map_in, fdentry->file);
		free(fdentry->file);
		free(fdentry);
		close(i->fd);
	} else {
		i->fdentry->refcount--;
	}
#else
	close(i->fd);
#endif

	if (i->free_buffer && i->buf != NULL) {
		free(i->buf);
	}
	free(i);

	in_list[id] = 0;
}

__attribute__ ((visibility("default")))
void bi_seek(int id, long offset, int length) {
	input_stream *restrict i = in_list[id];

	i->cur = sizeof(uint32_t) * offset;
	i->end = sizeof(uint32_t) * (offset + length);
	i->bp = i->B; // Load new data on next bi_next()

#ifdef REUSEFDS
	i->file_offset = i->cur;
#else
	if(lseek(i->fd, i->cur, SEEK_SET) == -1) {
		fprintf(stderr, "Unable to bi seek: %s\n", strerror(errno));
		exit(1);
	}
#endif
}

/*
 * OUTPUT STREAM OPERATIONS
 */

__attribute__ ((visibility("default")))
int bo_create(const char *restrict f, uint64_t start, int B, int dir) {
	int od = 0;

	od = stream_find_struct_storage((uintptr_t **)&out_list, &out_len);

	output_stream *restrict o = xmalloc(sizeof(output_stream));
	out_list[od] = o;

	// Allocate buffer on first use.
	o->buf = NULL;
	o->B = B;
	o->dir = dir;

	if (dir == STREAM_FORWARDS) {
		o->bp = 0;
	} else {
		o->bp = B;
	}

#ifdef REUSEFDS
	fdinfo *fdentry;

	// Create the map_out if it doesn't exist.
	if (map_out == NULL) {
		map_out = hashmap_new();
		if (map_out == NULL) {
			fprintf(stderr, "Error: Initialized map_out is null\n");
			exit(EXIT_FAILURE);
		}
	}

	// Find a file descriptor if we have one.
	int r = hashmap_get(map_out, (char *)f, (void **)&fdentry);

	if (r == MAP_MISSING) { // Element not found
		fdentry = xmalloc(sizeof(fdinfo));
		fdentry->fd       = open(f, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
		fdentry->refcount = 1;	
		fdentry->file     = strdup(f); // We need to store this.

		if(fdentry->fd < 0) {
			printf("Error in bo_create: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		r = hashmap_put(map_out, fdentry->file, fdentry);
		if (r == MAP_OMEM) {
			fprintf(stderr, "Out of memory adding to map_out\n");
			exit(EXIT_FAILURE);
		}
	} else {
		fdentry->refcount++;
	}

	o->fd          = fdentry->fd;
	o->fdentry     = fdentry;
#else
	o->fd = open(f, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);

	if(o->fd < 0) {
		printf("Error in bo_create: '%s'\n", strerror(errno));
		exit(EXIT_FAILURE);
	}
#endif

	o->free_buffer = true;
	lseek(o->fd, start*sizeof(uint32_t), SEEK_SET);

#ifdef REUSEFDS
	o->file_offset = start*sizeof(uint32_t);
#endif

	return od;
}

__attribute__ ((visibility("default")))
void bo_setbuffer(int od, int B, uint32_t *restrict buf) {
	output_stream *restrict o = out_list[od];

	assert(o->buf == NULL);

	o->B = B;

	if (o->dir == STREAM_FORWARDS) {
		o->bp = 0;
	} else {
		o->bp = B;
	}

	o->buf = buf;
	o->free_buffer = false;
}

__attribute__ ((visibility("default")))
void bo_write(int od, uint32_t v) {
	output_stream *restrict o = out_list[od];
	int r;

	assert(o->dir == STREAM_FORWARDS);

	if (o->buf == NULL) {
		o->buf = xcalloc(sizeof(uint32_t), o->B);
	}

	o->buf[o->bp] = v;
	o->bp++;

	if (o->bp == o->B) {

#ifdef REUSEFDS
		if (o->file_offset != lseek(o->fd, 0, SEEK_CUR)) {
			if (lseek(o->fd, o->file_offset, SEEK_SET) == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
			}
		}
#endif

		r = write(o->fd, o->buf, sizeof(uint32_t) * o->B);
		if(r == -1) {
			errorAndDie(6);
		}

#ifdef REUSEFDS
		o->file_offset = lseek(o->fd, 0, SEEK_CUR);
		if (o->file_offset == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
		}
#endif

		o->bp = 0;
	}
}

__attribute__ ((visibility("default")))
void bo_prepend(int od, uint32_t v) {
	output_stream *restrict o = out_list[od];
	int r;

	assert(o->dir == STREAM_BACKWARDS);

	if (o->buf == NULL) {
		o->buf = xcalloc(sizeof(uint32_t), o->B);
	}

	o->buf[--o->bp] = v;

	if (o->bp == 0) {

#ifdef REUSEFDS
		if (o->file_offset != lseek(o->fd, 0, SEEK_CUR)) {
			if (lseek(o->fd, o->file_offset, SEEK_SET) == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
			}
		}
#endif

		if (lseek(o->fd, -(o->B*sizeof(uint32_t)), SEEK_CUR) == -1) {
			printf("Error in bo_prepend1: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		r = write(o->fd, o->buf, sizeof(uint32_t) * o->B);
		if(r == -1) {
			errorAndDie(11);
		}

		if (lseek(o->fd, -(o->B*sizeof(uint32_t)), SEEK_CUR) == -1) {
			printf("Error in bo_prepend2: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

#ifdef REUSEFDS
		o->file_offset = lseek(o->fd, 0, SEEK_CUR);
		if (o->file_offset == -1) {
				fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
						__LINE__, __FILE__, strerror(errno));
				exit(EXIT_FAILURE);
		}
#endif

		o->bp = o->B;
	}
}

void __bo_flush(output_stream *restrict o) {
	int r;

#ifdef REUSEFDS
	if (o->file_offset != lseek(o->fd, 0, SEEK_CUR)) {
		if (lseek(o->fd, o->file_offset, SEEK_SET) == -1) {
			fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
					__LINE__, __FILE__, strerror(errno));
			exit(EXIT_FAILURE);
		}
	}
#endif

	// bp is above zero only if buf is initialized.
	if (o->dir == STREAM_FORWARDS && o->bp > 0) {
		r = write(o->fd, o->buf, sizeof(uint32_t) * o->bp);

		if(r == -1) {
			errorAndDie(7);
		}

		o->bp = 0;
	} else if (o->dir == STREAM_BACKWARDS && o->bp < o->B) {
		r = o->B - o->bp;

		if (lseek(o->fd, -(r*sizeof(uint32_t)), SEEK_CUR) == -1) {
			printf("Error in bo_flush: '%s'\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		r = write(o->fd, &o->buf[o->bp], sizeof(uint32_t) * r);
		if(r == -1) {
			errorAndDie(12);
		}
	}

#ifdef REUSEFDS
	o->file_offset = lseek(o->fd, 0, SEEK_CUR);
	if (o->file_offset == -1) {
			fprintf(stderr, "Unable to seek (%d, %s): %s\n", 
					__LINE__, __FILE__, strerror(errno));
			exit(EXIT_FAILURE);
	}
#endif
}

__attribute__ ((visibility("default")))
void bo_close(int od) {
	output_stream *restrict o = out_list[od];

	__bo_flush(o);

#ifdef REUSEFDS
	if (o->fdentry->refcount == 1) {
		fdinfo *fdentry = o->fdentry;
		hashmap_remove(map_out, fdentry->file);
		free(fdentry->file);
		free(fdentry);
		close(o->fd);
	} else {
		o->fdentry->refcount--;
	}
#else
	close(o->fd);
#endif

	if (o->free_buffer && o->buf != NULL) {
		free(o->buf);
	}

	free(o);

	out_list[od] = 0;
}

__attribute__ ((visibility("default")))
void bo_seek(int od, long offset) {
	output_stream *restrict o = out_list[od];

	__bo_flush(o);

#ifdef REUSEFDS
	o->fd = offset*sizeof(uint32_t);
#else
	if(lseek(o->fd, offset*sizeof(uint32_t), SEEK_SET) == -1) {
		fprintf(stderr, "Unable to bo seek: %s\n", strerror(errno));
		exit(1);
	}
#endif
}
