// Singleton to handle accesses to the api.
var CompanyStorage = {
  fetchAll: function() {
    return axios.get('/api/companies');
  },

  fetch: function(id) {
    return axios.get('/api/companies/' + id);
  },

  create: function(company) {
    return axios.post('/api/companies', company);
  },

  delete: function(company) {
    return axios.delete('/api/companies/' + company.id);
  },

  nilCompany: function() {
    return {
      name: '',
      cvr: '',
      address: '',
      city: '',
      country: '',
      phone: '',
    }
  },
};

var State = {
  View:   1,
  Create: 2,
};

// Each row in the list is a <company>.
var Company = {
  props: ['company'],

  data: function() {
    return {}
  },

  template: "#listitemtemplate",

  methods: {
    deleteAction: function() {
      this.$emit('delete');
    },

    showAction: function() {
      this.$emit('show');
    },
  }
};

// Main Appllication
var app = new Vue({
    el: '#content',
    template: '#contenttemplate',

    data: {
      // Company data.
      companies: [],

      // Current company shown in form.
      currentCompany: CompanyStorage.nilCompany(),

      // State of the form.
      state: State.View,

      // Filter for search bar.
      filter: '',
      
      // This is assigned the error description returned when trying to create
      // a company. Each input then looks for warnings[name].
      warnings: {},
    },

    beforeMount: function() {
      // Fetch data before we mount.
      CompanyStorage.fetchAll().then(function(response) {
          // A bit tricky, apparently.. Line [0] adds the entire
          // array as a single element. So does [1]..
          // [0]: Array.prototype.splice.apply(app.companies, [-1,0].concat(response.data))
          // [1]: app.companies.splice(-1, 0, response.data);
          if (response.data !== null && response.data.length > 0) {
            app.companies.splice(-1, 0, ...response.data);

            app.companies.sort(function(a, b) {
              return b.id - a.id;
            });

          }
        })
        .catch(function(error) {
          console.log(error);
        });
    },

    components: {
      'company': Company
    },

    computed: {
      companiesFiltered: function() {
        return this.companies.filter(function(company) {
          return company.name.indexOf(app.filter) !== -1
        }).sort(function(a, b) {
          return a.id < b.id;
        });
      },
    },

    methods: {
      // On delete click of <li>
      deleteCompany: function(company) {
        CompanyStorage.delete(company).then(function() {
            let idx = app.companies.indexOf(company)
            app.companies.splice(idx, 1);
        })
      },

      // On click on <li>
      showCompany: function(company) {
        if (this.state === State.View) {
          CompanyStorage.fetch(company.id).then(function(resp) {
            app.currentCompany = resp.data;
          }).catch(function() {
          });
        }
      },

      // On create button
      beginCreate: function() {
        this.currentCompany = CompanyStorage.nilCompany();
        this.state = State.Create;
      },

      // On cancel button
      cancelCreate: function() {
        this.currentCompany = CompanyStorage.nilCompany();
        this.state = State.View;
        app.warnings = {};
      },

      submitCreated: function() {
        CompanyStorage.create(this.currentCompany)
          .then(function(response) {
            app.companies.unshift(response.data);
            app.state = State.View;
            app.warnings = {};
          }).catch(function(error) {
            app.warnings = error.response.data;
          });
      },

      getWarnings: function(key) {
        if (this.warnings.hasOwnProperty(key)) {
          return this.warnings[key].join("\n");
        }
      }
    },
});
